<?php declare(strict_types=1);

namespace ResTelae\Gbif\Tests\Unit;

use PHPUnit\Framework\TestCase;
use ResTelae\Gbif\Species;

final class SpeciesTest extends TestCase
{
    /** @var Species */
    private $species;

    protected function setUp(): void
    {
        parent::setUp();

        $this->species = new Species();
    }

    public function testShouldThrowExceptionIfInvalidDataChoice()
    {
        $this->expectExceptionMessage('Illegal choice for `data`');
        $this->species->nameUsage([], 'invalid');

        $this->fail("No exception raised");
    }

    public function testShouldThrowExceptionIfNoDataKeyProvided()
    {
        $this->expectExceptionMessage('You must specify a key if `data` does not equal `all`');
        $this->species->nameUsage([], 'name');

        $this->fail("No exception raised");
    }

    public function testShouldThrowExceptionOnRootDataIfNoUuidOrShortNameProvided()
    {
        $this->expectExceptionMessage('`uuid` and `short_name` cannot be both NULL if `data` equals "root"');
        $this->species->nameUsage([], 'root', '123');

        $this->fail("No exception raised");
    }

    public function testShouldGetValidResponseWithValidArguments()
    {
        $response = $this->species->nameUsage([]);

        $this->assertEquals(0, $response['offset']);
        $this->assertEquals(100, $response['limit']);
        $this->assertFalse($response['endOfRecords']);
        $this->assertIsArray($response['results']);
    }

    public function testShouldGetValidResponseFromNameUsageByKey()
    {
        $response = $this->species->nameUsageByKey(5231190);

        $this->assertEquals(5231190, $response['key']);
        $this->assertEquals('Animalia', $response['kingdom']);
        $this->assertEquals('Passer domesticus', $response['species']);
    }

    public function testShouldGetValidResponseFromNameBackbone()
    {
        $response = $this->species->nameBackbone(
            [
                'verbose' => true,
                'kingdom' => 'Plantae',
                'name'    => 'Oenante',
            ]
        );

        $this->assertEquals(3034893, $response['usageKey']);
        $this->assertEquals('Oenanthe', $response['canonicalName']);
        $this->assertEquals('GENUS', $response['rank']);
    }

    public function testShouldGetValidResponseFromNameLookup()
    {
        $response = $this->species->nameLookup(
            [
                'q'      => 'Puma',
                'rank'   => 'GENUS',
                'offset' => 0,
                'limit'  => 20,
            ]
        );

        $this->assertEquals(0, $response['offset']);
        $this->assertEquals(20, $response['limit']);
        $this->assertIsArray($response['results']);
        $this->assertEquals('Puma', $response['results'][0]['genus']);
        $this->assertEquals('Puma', $response['results'][0]['canonicalName']);
    }

    public function testShouldGetValidResponseFromNameSuggest()
    {
        $response = $this->species->nameSuggest(
            [
                'q'      => 'Pum',
                'rank'   => 'GENUS',
                'offset' => 0,
                'limit'  => 20,
            ]
        );

        $this->assertIsArray($response);
        $this->assertCount(20, $response);
        $this->assertEquals('Pumililema', $response[0]['genus']);
        $this->assertEquals('Pumililema', $response[0]['canonicalName']);
    }
}
