<?php

namespace ResTelae\Gbif;

use Guzzle\Http\Exception\RequestException;
use GuzzleHttp\Client;

/**
 * GBIF: generic class.
 */
abstract class Gbif {

  const GBIF_BASEURL = 'https://api.gbif.org/v1/';

  /**
   * HTTP client.
   *
   * @var \GuzzleHttp\Client
   */
  protected $gbifClient;

  /**
   * Constructs a GBIF object.
   */
  public function __construct() {
    $this->gbifClient = new Client([
      'timeout' => 30,
      'base_uri' => self::GBIF_BASEURL,
    ]);
  }

  /**
   * Sends an HTTP GET request to GBIF.
   *
   * @param string $uri
   *   URI without base URI and leading slash.
   * @param array $args
   *   Arguments to be passed through the query string.
   *
   * @return array
   *   An array of results.
   */
  protected function gbifGet($uri, array $args = []) {
    try {
      $response = $this->gbifClient->get($uri, ['query' => $args]);
    }
    catch (RequestException $e) {
      error_log((string) $e->getRequest());
      if ($e->hasResponse) {
        error_log((string) $e->getResponse());
      }
      return [];
    }

    $data = (string) $response->getBody();
    return json_decode($data, TRUE);
  }

  /**
   * Represents boolean value as literal.
   *
   * @param bool $val
   *   Boolean value to convert.
   *
   * @return string
   *   Literal representation ('true' or 'false').
   */
  protected function bool2str(bool $val) {
    return var_export($val, TRUE);
  }

}
